   /*******************************************************/
   /*      "C" Language Integrated Production System      */
   /*                                                     */
   /*             CLIPS Version 6.30  10/19/06            */
   /*                                                     */
   /*                RULE COMMANDS MODULE                 */
   /*******************************************************/

/*************************************************************/
/* Purpose: Provides the matches command. Also provides the  */
/*   the developer commands show-joins and rule-complexity.  */
/*   Also provides the initialization routine which          */
/*   registers rule commands found in other modules.         */
/*                                                           */
/* Principal Programmer(s):                                  */
/*      Gary D. Riley                                        */
/*                                                           */
/* Contributing Programmer(s):                               */
/*                                                           */
/* Revision History:                                         */
/*                                                           */
/*      6.24: Removed CONFLICT_RESOLUTION_STRATEGIES         */
/*            INCREMENTAL_RESET, and LOGICAL_DEPENDENCIES    */
/*            compilation flags.                             */
/*                                                           */
/*            Renamed BOOLEAN macro type to intBool.         */
/*                                                           */
/*      6.30: Added support for hashed alpha memories.       */
/*                                                           */
/*            Added matches-count function.                  */
/*                                                           */
/*            Added get-join-hashing and set-join-hashing    */
/*            functions.                                     */
/*                                                           */
/*************************************************************/

#define _RULECOM_SOURCE_

#include <stdio.h>
#define _STDIO_INCLUDED_
#include <string.h>

#include "setup.h"

#if DEFRULE_CONSTRUCT

#include "argacces.h"
#include "constant.h"
#include "constrct.h"
#include "crstrtgy.h"
#include "engine.h"
#include "envrnmnt.h"
#include "evaluatn.h"
#include "extnfunc.h"
#include "incrrset.h"
#include "lgcldpnd.h"
#include "memalloc.h"
#include "pattern.h"
#include "reteutil.h"
#include "router.h"
#include "ruledlt.h"
#include "watch.h"

#if BLOAD || BLOAD_AND_BSAVE || BLOAD_ONLY
#include "rulebin.h"
#endif

#include "rulecom.h"

/***************************************/
/* LOCAL INTERNAL FUNCTION DEFINITIONS */
/***************************************/

#if DEVELOPER
   static void                    ShowJoins(void *,void *);
#endif

/****************************************************************/
/* DefruleCommands: Initializes defrule commands and functions. */
/****************************************************************/
globle void DefruleCommands(
  void *theEnv)
  {
#if ! RUN_TIME
   EnvDefineFunction2(theEnv,"run",'v', PTIEF RunCommand,"RunCommand", "*1i");
   EnvDefineFunction2(theEnv,"halt",'v', PTIEF HaltCommand,"HaltCommand","00");
   EnvDefineFunction2(theEnv,"focus",'b', PTIEF FocusCommand,"FocusCommand", "1*w");
   EnvDefineFunction2(theEnv,"clear-focus-stack",'v',PTIEF ClearFocusStackCommand,
                                       "ClearFocusStackCommand","00");
   EnvDefineFunction2(theEnv,"get-focus-stack",'m',PTIEF GetFocusStackFunction,
                                     "GetFocusStackFunction","00");
   EnvDefineFunction2(theEnv,"pop-focus",'w',PTIEF PopFocusFunction,
                               "PopFocusFunction","00");
   EnvDefineFunction2(theEnv,"get-focus",'w',PTIEF GetFocusFunction,
                               "GetFocusFunction","00");
#if DEBUGGING_FUNCTIONS
   EnvDefineFunction2(theEnv,"set-break",'v', PTIEF SetBreakCommand,
                               "SetBreakCommand","11w");
   EnvDefineFunction2(theEnv,"remove-break",'v', PTIEF RemoveBreakCommand,
                                  "RemoveBreakCommand", "*1w");
   EnvDefineFunction2(theEnv,"show-breaks",'v', PTIEF ShowBreaksCommand,
                                 "ShowBreaksCommand", "01w");
   EnvDefineFunction2(theEnv,"matches",'v',PTIEF MatchesCommand,"MatchesCommand","11w");
   EnvDefineFunction2(theEnv,"matches-count",'v',PTIEF MatchesCountCommand,"MatchesCountCommand","11w");
   EnvDefineFunction2(theEnv,"list-focus-stack",'v', PTIEF ListFocusStackCommand,
                                      "ListFocusStackCommand", "00");
   EnvDefineFunction2(theEnv,"dependencies", 'v', PTIEF DependenciesCommand,
                                   "DependenciesCommand", "11h");
   EnvDefineFunction2(theEnv,"dependents",   'v', PTIEF DependentsCommand,
                                   "DependentsCommand", "11h");
#endif /* DEBUGGING_FUNCTIONS */

   EnvDefineFunction2(theEnv,"get-incremental-reset",'b',
                   GetIncrementalResetCommand,"GetIncrementalResetCommand","00");
   EnvDefineFunction2(theEnv,"set-incremental-reset",'b',
                   SetIncrementalResetCommand,"SetIncrementalResetCommand","11");

   EnvDefineFunction2(theEnv,"get-join-hashing",'b',
                   GetJoinHashingCommand,"GetJoinHashingCommand","00");
   EnvDefineFunction2(theEnv,"set-join-hashing",'b',
                   SetJoinHashingCommand,"SetJoinHashingCommand","11");

   EnvDefineFunction2(theEnv,"get-strategy", 'w', PTIEF GetStrategyCommand,  "GetStrategyCommand", "00");
   EnvDefineFunction2(theEnv,"set-strategy", 'w', PTIEF SetStrategyCommand,  "SetStrategyCommand", "11w");

#if DEVELOPER && (! BLOAD_ONLY)
   EnvDefineFunction2(theEnv,"rule-complexity",'l', PTIEF RuleComplexityCommand,"RuleComplexityCommand", "11w");
   EnvDefineFunction2(theEnv,"show-joins",   'v', PTIEF ShowJoinsCommand,    "ShowJoinsCommand", "11w");
   EnvDefineFunction2(theEnv,"show-aht",   'v', PTIEF ShowAlphaHashTable,    "ShowAlphaHashTable", "00");
#if DEBUGGING_FUNCTIONS
   AddWatchItem(theEnv,"rule-analysis",0,&DefruleData(theEnv)->WatchRuleAnalysis,0,NULL,NULL);
#endif
#endif /* DEVELOPER && (! BLOAD_ONLY) */

#else
#if MAC_MCW || IBM_MCW || MAC_XCD
#pragma unused(theEnv)
#endif
#endif /* ! RUN_TIME */
  }

/***************************************/
/* EnvGetJoinHashing: C access routine */
/*   for the get-join-hashing command. */
/***************************************/
globle intBool EnvGetJoinHashing(
  void *theEnv)
  {   
   return(DefruleData(theEnv)->JoinHashingFlag);
  }

/***************************************/
/* EnvSetJoinHashing: C access routine */
/*   for the set-join-hashing command. */
/***************************************/
globle intBool EnvSetJoinHashing(
  void *theEnv,
  int value)
  {
   int ov;
   struct defmodule *theModule;

   ov = DefruleData(theEnv)->JoinHashingFlag;

   SaveCurrentModule(theEnv);

   for (theModule = (struct defmodule *) EnvGetNextDefmodule(theEnv,NULL);
        theModule != NULL;
        theModule = (struct defmodule *) EnvGetNextDefmodule(theEnv,theModule))
     {
      EnvSetCurrentModule(theEnv,(void *) theModule);
      if (EnvGetNextDefrule(theEnv,NULL) != NULL)
        {
         RestoreCurrentModule(theEnv);
         return(-1);
        }
     }
     
   RestoreCurrentModule(theEnv);

   DefruleData(theEnv)->JoinHashingFlag = value;
   return(ov);
  }

/*********************************************/
/* SetJoinHashingCommand: H/L access routine */
/*   for the set-join-hashing command.       */
/*********************************************/
globle int SetJoinHashingCommand(
  void *theEnv)
  {
   int oldValue;
   DATA_OBJECT argPtr;
   struct defmodule *theModule;

   oldValue = EnvGetJoinHashing(theEnv);

   /*============================================*/
   /* Check for the correct number of arguments. */
   /*============================================*/

   if (EnvArgCountCheck(theEnv,"set-join-hashing",EXACTLY,1) == -1)
     { return(oldValue); }

   /*=====================================*/
   /* The alpha hashing behavior can't be */
   /* changed when rules are loaded.      */
   /*=====================================*/

   SaveCurrentModule(theEnv);

   for (theModule = (struct defmodule *) EnvGetNextDefmodule(theEnv,NULL);
        theModule != NULL;
        theModule = (struct defmodule *) EnvGetNextDefmodule(theEnv,theModule))
     {
      EnvSetCurrentModule(theEnv,(void *) theModule);
      if (EnvGetNextDefrule(theEnv,NULL) != NULL)
        {
         RestoreCurrentModule(theEnv);
         PrintErrorID(theEnv,"RULECOM",1,FALSE);
         EnvPrintRouter(theEnv,WERROR,"The join hashing behavior cannot be changed with rules loaded.\n");
         SetEvaluationError(theEnv,TRUE);
         return(oldValue);
        }
     }
     
   RestoreCurrentModule(theEnv);

   /*==============================================*/
   /* The symbol FALSE disables alpha hashing. Any */
   /* other value enables alpha hashing.           */
   /*==============================================*/

   EnvRtnUnknown(theEnv,1,&argPtr);

   if ((argPtr.value == EnvFalseSymbol(theEnv)) && (argPtr.type == SYMBOL))
     { EnvSetJoinHashing(theEnv,FALSE); }
   else
     { EnvSetJoinHashing(theEnv,TRUE); }

   /*=======================*/
   /* Return the old value. */
   /*=======================*/

   return(oldValue);
  }

/*********************************************/
/* GetJoinHashingCommand: H/L access routine */
/*   for the get-join-hashing command.       */
/*********************************************/
globle int GetJoinHashingCommand(
  void *theEnv)
  {
   int oldValue;

   oldValue = EnvGetJoinHashing(theEnv);

   if (EnvArgCountCheck(theEnv,"get-join-hashing",EXACTLY,0) == -1)
     { return(oldValue); }

   return(oldValue);
  }

#if DEBUGGING_FUNCTIONS

/****************************************/
/* MatchesCommand: H/L access routine   */
/*   for the matches command.           */
/****************************************/
globle void MatchesCommand(
  void *theEnv)
  {
   char *ruleName;
   void *rulePtr;

   ruleName = GetConstructName(theEnv,"matches","rule name");
   if (ruleName == NULL) return;

   rulePtr = EnvFindDefrule(theEnv,ruleName);
   if (rulePtr == NULL)
     {
      CantFindItemErrorMessage(theEnv,"defrule",ruleName);
      return;
     }

   EnvMatches(theEnv,rulePtr);
  }

/********************************/
/* EnvMatches: C access routine */
/*   for the matches command.   */
/********************************/
globle intBool EnvMatches(
  void *theEnv,
  void *theRule)
  {
   struct defrule *rulePtr, *tmpPtr;
   struct partialMatch *listOfMatches, **theStorage;
   struct alphaMemoryHash *listOfHashNodes, **theAlphaStorage;
   struct joinNode *theJoin, *lastJoin;
   int i, depth;
   ACTIVATION *agendaPtr;
   int flag;
   int matchesDisplayed;

   /*=================================================*/
   /* Loop through each of the disjuncts for the rule */
   /*=================================================*/

   for (rulePtr = (struct defrule *) theRule, tmpPtr = rulePtr;
        rulePtr != NULL;
        rulePtr = rulePtr->disjunct)
     {
      /*======================================*/
      /* Determine the last join in the rule. */
      /*======================================*/

      lastJoin = rulePtr->lastJoin;

      /*===================================*/
      /* Determine the number of patterns. */
      /*===================================*/

      depth = GetPatternNumberFromJoin(lastJoin);

      /*=========================================*/
      /* Store the alpha memory partial matches. */
      /*=========================================*/

      theAlphaStorage = (struct alphaMemoryHash **)
                        genalloc(theEnv,(unsigned) (depth * sizeof(struct alphaMemoryHash *)));

      theJoin = lastJoin;
      i = depth - 1;
      while (theJoin != NULL)
        {
         if (theJoin->joinFromTheRight)
           { theJoin = (struct joinNode *) theJoin->rightSideEntryStructure; }
         else
           {
            theAlphaStorage[i] = ((struct patternNodeHeader *) theJoin->rightSideEntryStructure)->firstHash;  /* TBD Alpha Memory Hashing */
            i--;
            theJoin = theJoin->lastLevel;
           }
        }

      /*========================================*/
      /* List the alpha memory partial matches. */
      /*========================================*/

      for (i = 0; i < depth; i++)
        {
         if (GetHaltExecution(theEnv) == TRUE)
           {
            genfree(theEnv,theAlphaStorage,(unsigned) (depth * sizeof(struct alphaMemoryHash *)));
            return(TRUE);
           }

         EnvPrintRouter(theEnv,WDISPLAY,"Matches for Pattern ");
         PrintLongInteger(theEnv,WDISPLAY,(long int) i + 1);
         EnvPrintRouter(theEnv,WDISPLAY,"\n");

         flag = 1;
         for (listOfHashNodes = theAlphaStorage[i];
              listOfHashNodes != NULL;
              listOfHashNodes = listOfHashNodes->nextHash)
           {
            listOfMatches = listOfHashNodes->alphaMemory;

            while (listOfMatches != NULL)
              {
               if (GetHaltExecution(theEnv) == TRUE)
                 {
                  genfree(theEnv,theAlphaStorage,(unsigned) (depth * sizeof(struct alphaMemoryHash *)));
                  return(TRUE);
                 }
                 
               flag = 0;
               PrintPartialMatch(theEnv,WDISPLAY,listOfMatches);
               EnvPrintRouter(theEnv,WDISPLAY,"\n");
               listOfMatches = listOfMatches->nextInNode;
              }
           }
           
         if (flag) EnvPrintRouter(theEnv,WDISPLAY," None\n");
        }

      genfree(theEnv,theAlphaStorage,(unsigned) (depth * sizeof(struct alphaMemoryHash *)));

      /*========================================*/
      /* Store the beta memory partial matches. */
      /*========================================*/

      depth = lastJoin->depth;
      theStorage = (struct partialMatch **) genalloc(theEnv,(unsigned) (depth * sizeof(struct partialMatch)));

      theJoin = lastJoin;
      for (i = depth - 1; i >= 0; i--)
        {
         theStorage[i] = theJoin->beta;
         theJoin = theJoin->lastLevel;
        }

      /*=======================================*/
      /* List the beta memory partial matches. */
      /*=======================================*/

      for (i = 1; i < depth; i++)
        {
         if (GetHaltExecution(theEnv) == TRUE)
           {
            genfree(theEnv,theStorage,(unsigned) (depth * sizeof(struct partialMatch)));
            return(TRUE);
           }

         matchesDisplayed = 0;
         EnvPrintRouter(theEnv,WDISPLAY,"Partial matches for CEs 1 - ");
         PrintLongInteger(theEnv,WDISPLAY,(long int) i + 1);
         EnvPrintRouter(theEnv,WDISPLAY,"\n");
         listOfMatches = theStorage[i];

         while (listOfMatches != NULL)
           {
            if (GetHaltExecution(theEnv) == TRUE)
              {
               genfree(theEnv,theStorage,(unsigned) (depth * sizeof(struct partialMatch)));
               return(TRUE);
              }

            if (listOfMatches->counterf == FALSE)
              {
               matchesDisplayed++;
               PrintPartialMatch(theEnv,WDISPLAY,listOfMatches);
               EnvPrintRouter(theEnv,WDISPLAY,"\n");
              }
            listOfMatches = listOfMatches->nextInNode;
           }

         if (matchesDisplayed == 0) { EnvPrintRouter(theEnv,WDISPLAY," None\n"); }
        }

      genfree(theEnv,theStorage,(unsigned) (depth * sizeof(struct partialMatch)));
     }

   /*===================*/
   /* List activations. */
   /*===================*/

   rulePtr = tmpPtr;
   EnvPrintRouter(theEnv,WDISPLAY,"Activations\n");
   flag = 1;
   for (agendaPtr = (struct activation *) EnvGetNextActivation(theEnv,NULL);
        agendaPtr != NULL;
        agendaPtr = (struct activation *) EnvGetNextActivation(theEnv,agendaPtr))
     {
      if (GetHaltExecution(theEnv) == TRUE) return(TRUE);

      if (((struct activation *) agendaPtr)->theRule->header.name == rulePtr->header.name)
        {
         flag = 0;
         PrintPartialMatch(theEnv,WDISPLAY,GetActivationBasis(agendaPtr));
         EnvPrintRouter(theEnv,WDISPLAY,"\n");
        }
     }

   if (flag) EnvPrintRouter(theEnv,WDISPLAY," None\n");

   return(TRUE);
  }

/*******************************************/
/* MatchesCountCommand: H/L access routine */
/*   for the matches-count command.        */
/*******************************************/
globle void MatchesCountCommand(
  void *theEnv)
  {
   char *ruleName;
   void *rulePtr;

   ruleName = GetConstructName(theEnv,"matches-count","rule name");
   if (ruleName == NULL) return;

   rulePtr = EnvFindDefrule(theEnv,ruleName);
   if (rulePtr == NULL)
     {
      CantFindItemErrorMessage(theEnv,"defrule",ruleName);
      return;
     }

   EnvMatchesCount(theEnv,rulePtr);
  }

/*************************************/
/* EnvMatchesCount: C access routine */
/*   for the matches-count command.  */
/*************************************/
globle intBool EnvMatchesCount(
  void *theEnv,
  void *theRule)
  {
   struct defrule *rulePtr, *tmpPtr;
   struct partialMatch *listOfMatches, **theStorage;
   struct alphaMemoryHash *listOfHashNodes, **theAlphaStorage;
   struct joinNode *theJoin, *lastJoin;
   int i, depth;
   ACTIVATION *agendaPtr;
   long count;
#if DEVELOPER
   long count2;
#endif

   /*=================================================*/
   /* Loop through each of the disjuncts for the rule */
   /*=================================================*/

   for (rulePtr = (struct defrule *) theRule, tmpPtr = rulePtr;
        rulePtr != NULL;
        rulePtr = rulePtr->disjunct)
     {
      /*======================================*/
      /* Determine the last join in the rule. */
      /*======================================*/

      lastJoin = rulePtr->lastJoin;

      /*===================================*/
      /* Determine the number of patterns. */
      /*===================================*/

      depth = GetPatternNumberFromJoin(lastJoin);

      /*=========================================*/
      /* Store the alpha memory partial matches. */
      /*=========================================*/

      theAlphaStorage = (struct alphaMemoryHash **)
                        genalloc(theEnv,(unsigned) (depth * sizeof(struct alphaMemoryHash *)));

      theJoin = lastJoin;
      i = depth - 1;
      while (theJoin != NULL)
        {
         if (theJoin->joinFromTheRight)
           { theJoin = (struct joinNode *) theJoin->rightSideEntryStructure; }
         else
           {
            theAlphaStorage[i] = ((struct patternNodeHeader *) theJoin->rightSideEntryStructure)->firstHash;
            i--;
            theJoin = theJoin->lastLevel;
           }
        }

      /*========================================*/
      /* List the alpha memory partial matches. */
      /*========================================*/

      for (i = 0; i < depth; i++)
        {
         if (GetHaltExecution(theEnv) == TRUE)
           {
            genfree(theEnv,theAlphaStorage,(unsigned) (depth * sizeof(struct alphaMemoryHash *)));
            return(TRUE);
           }

         EnvPrintRouter(theEnv,WDISPLAY,"Matches for Pattern ");
         PrintLongInteger(theEnv,WDISPLAY,(long int) i + 1);
         EnvPrintRouter(theEnv,WDISPLAY,": ");

         count = 0;
         for (listOfHashNodes = theAlphaStorage[i];
              listOfHashNodes != NULL;
              listOfHashNodes = listOfHashNodes->nextHash)
           {
            listOfMatches = listOfHashNodes->alphaMemory;

            while (listOfMatches != NULL)
              {
               if (GetHaltExecution(theEnv) == TRUE)
                 {
                  genfree(theEnv,theAlphaStorage,(unsigned) (depth * sizeof(struct alphaMemoryHash *)));
                  return(TRUE);
                 }
                 
               count++;
               listOfMatches = listOfMatches->nextInNode;
              }
           }
           
         PrintLongInteger(theEnv,WDISPLAY,count);
         EnvPrintRouter(theEnv,WDISPLAY,"\n");
        }

      genfree(theEnv,theAlphaStorage,(unsigned) (depth * sizeof(struct alphaMemoryHash *)));

      /*========================================*/
      /* Store the beta memory partial matches. */
      /*========================================*/

      depth = lastJoin->depth;
      theStorage = (struct partialMatch **) genalloc(theEnv,(unsigned) (depth * sizeof(struct partialMatch)));

      theJoin = lastJoin;
      for (i = depth - 1; i >= 0; i--)
        {
         theStorage[i] = theJoin->beta;
         theJoin = theJoin->lastLevel;
        }

      /*=======================================*/
      /* List the beta memory partial matches. */
      /*=======================================*/

      for (i = 1; i < depth; i++)
        {
         if (GetHaltExecution(theEnv) == TRUE)
           {
            genfree(theEnv,theStorage,(unsigned) (depth * sizeof(struct partialMatch)));
            return(TRUE);
           }

         count = 0;
#if DEVELOPER
         count2 = 0;
#endif
         EnvPrintRouter(theEnv,WDISPLAY,"Partial matches for CEs 1 - ");
         PrintLongInteger(theEnv,WDISPLAY,(long int) i + 1);
         EnvPrintRouter(theEnv,WDISPLAY,": ");
         listOfMatches = theStorage[i];

         while (listOfMatches != NULL)
           {
            if (GetHaltExecution(theEnv) == TRUE)
              {
               genfree(theEnv,theStorage,(unsigned) (depth * sizeof(struct partialMatch)));
               return(TRUE);
              }

#if DEVELOPER
            count2++;
#endif
            if (listOfMatches->counterf == FALSE)
              { count++; }
            listOfMatches = listOfMatches->nextInNode;
           }

         PrintLongInteger(theEnv,WDISPLAY,count);
#if DEVELOPER
         EnvPrintRouter(theEnv,WDISPLAY,"/"); 
         PrintLongInteger(theEnv,WDISPLAY,count2);
#endif
         EnvPrintRouter(theEnv,WDISPLAY,"\n"); 
        }

      genfree(theEnv,theStorage,(unsigned) (depth * sizeof(struct partialMatch)));
     }

   /*===================*/
   /* List activations. */
   /*===================*/

   rulePtr = tmpPtr;
   EnvPrintRouter(theEnv,WDISPLAY,"Activations: ");
   count = 0;
   for (agendaPtr = (struct activation *) EnvGetNextActivation(theEnv,NULL);
        agendaPtr != NULL;
        agendaPtr = (struct activation *) EnvGetNextActivation(theEnv,agendaPtr))
     {
      if (GetHaltExecution(theEnv) == TRUE) return(TRUE);

      if (((struct activation *) agendaPtr)->theRule->header.name == rulePtr->header.name)
        { count++; }
     }

   PrintLongInteger(theEnv,WDISPLAY,count);
   EnvPrintRouter(theEnv,WDISPLAY,"\n");

   return(TRUE);
  }

#endif /* DEBUGGING_FUNCTIONS */

#if DEVELOPER
/***********************************************/
/* RuleComplexityCommand: H/L access routine   */
/*   for the rule-complexity function.         */
/***********************************************/
globle long RuleComplexityCommand(
  void *theEnv)
  {
   char *ruleName;
   struct defrule *rulePtr;

   ruleName = GetConstructName(theEnv,"rule-complexity","rule name");
   if (ruleName == NULL) return(-1);

   rulePtr = (struct defrule *) EnvFindDefrule(theEnv,ruleName);
   if (rulePtr == NULL)
     {
      CantFindItemErrorMessage(theEnv,"defrule",ruleName);
      return(-1);
     }

   return(rulePtr->complexity);
  }

/******************************************/
/* ShowJoinsCommand: H/L access routine   */
/*   for the show-joins command.          */
/******************************************/
globle void ShowJoinsCommand(
  void *theEnv)
  {
   char *ruleName;
   void *rulePtr;

   ruleName = GetConstructName(theEnv,"show-joins","rule name");
   if (ruleName == NULL) return;

   rulePtr = EnvFindDefrule(theEnv,ruleName);
   if (rulePtr == NULL)
     {
      CantFindItemErrorMessage(theEnv,"defrule",ruleName);
      return;
     }

   ShowJoins(theEnv,rulePtr);

   return;
  }

/*********************************/
/* ShowJoins: C access routine   */
/*   for the show-joins command. */
/*********************************/
static void ShowJoins(
  void *theEnv,
  void *theRule)
  {
   struct defrule *rulePtr;
   struct joinNode *theJoin;
   struct joinNode *joinList[MAXIMUM_NUMBER_OF_PATTERNS];
   int numberOfJoins;

   rulePtr = (struct defrule *) theRule;

   /*=================================================*/
   /* Loop through each of the disjuncts for the rule */
   /*=================================================*/

   while (rulePtr != NULL)
     {
      /*=====================================*/
      /* Determine the number of join nodes. */
      /*=====================================*/

      numberOfJoins = -1;
      theJoin = rulePtr->lastJoin;
      while (theJoin != NULL)
        {
         if (theJoin->joinFromTheRight)
           { theJoin = (struct joinNode *) theJoin->rightSideEntryStructure; }
         else
           {
            numberOfJoins++;
            joinList[numberOfJoins] = theJoin;
            theJoin = theJoin->lastLevel;
           }
        }

      /*====================*/
      /* Display the joins. */
      /*====================*/

      while (numberOfJoins >= 0)
        {
         char buffer[20];
         sprintf(buffer,"%2d%c%c: ",(int) joinList[numberOfJoins]->depth,
                                     (joinList[numberOfJoins]->patternIsNegated) ? 'n' : ' ',
                                     (joinList[numberOfJoins]->logicalJoin) ? 'l' : ' ');
         EnvPrintRouter(theEnv,WDISPLAY,buffer);
         PrintExpression(theEnv,WDISPLAY,joinList[numberOfJoins]->networkTest);
         EnvPrintRouter(theEnv,WDISPLAY,"\n");
         
         if (joinList[numberOfJoins]->leftHash != NULL)
           {
            EnvPrintRouter(theEnv,WDISPLAY,"    LH : ");
            PrintExpression(theEnv,WDISPLAY,joinList[numberOfJoins]->leftHash);
            EnvPrintRouter(theEnv,WDISPLAY,"\n");
           }
         
         numberOfJoins--;
        };

      /*===============================*/
      /* Proceed to the next disjunct. */
      /*===============================*/

      rulePtr = rulePtr->disjunct;
      if (rulePtr != NULL) EnvPrintRouter(theEnv,WDISPLAY,"\n");
     }
  }

/******************************************************/
/* ShowAlphaHashTable: Displays the number of entries */
/*   in each slot of the alpha hash table.            */
/******************************************************/
globle void ShowAlphaHashTable(
   void *theEnv)
   {
    int i, count;
    long totalCount = 0;
    struct alphaMemoryHash *theEntry;
    struct partialMatch *theMatch;
    char buffer[40];

    for (i = 0; i < ALPHA_MEMORY_HASH_SIZE; i++)
      {
       for (theEntry =  DefruleData(theEnv)->AlphaMemoryTable[i], count = 0;
            theEntry != NULL;
            theEntry = theEntry->next)
         { count++; }

       if (count != 0)
         {
          totalCount += count;
          sprintf(buffer,"%4d: %4d ->",i,count);
          EnvPrintRouter(theEnv,WDISPLAY,buffer);
          
          for (theEntry =  DefruleData(theEnv)->AlphaMemoryTable[i], count = 0;
               theEntry != NULL;
               theEntry = theEntry->next)
            {
             for (theMatch = theEntry->alphaMemory;
                  theMatch != NULL;
                  theMatch = theMatch->nextInNode)
               { count++; }
               
             sprintf(buffer," %4d",count);
             EnvPrintRouter(theEnv,WDISPLAY,buffer);
             if (theEntry->owner->rightHash == NULL)
               { EnvPrintRouter(theEnv,WDISPLAY,"*"); }
               /*
             if (count > 5)
               {
                for (theMatch = theEntry->alphaMemory;
                     theMatch != NULL;
                     theMatch = theMatch->nextInNode)
                  { 
                   EnvPrintRouter(theEnv,WDISPLAY," ");
                   PrintPartialMatch(theEnv,WDISPLAY,theMatch); 
                  }
               }
               */
            }
          
          EnvPrintRouter(theEnv,WDISPLAY,"\n");
         }
      }
    sprintf(buffer,"Total Count: %ld\n",totalCount);
    EnvPrintRouter(theEnv,WDISPLAY,buffer);
   }

#endif /* DEVELOPER */

#endif /* DEFRULE_CONSTRUCT */

